// extension/offscreen/offscreen.ts
class OffscreenCardAdder {
  iframe = null;
  currentCard = null;
  retryCount = 0;
  maxRetries = 3;
  constructor() {
    this.setupMessageListener();
  }
  setupMessageListener() {
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      if (request.target !== "offscreen")
        return;
      console.log("[Offscreen] Received message:", request.type);
      switch (request.type) {
        case "ADD_CARD_TO_AIRBNB":
          this.handleAddCard(request.data).then(sendResponse).catch((error) => {
            console.error("[Offscreen] Error:", error);
            sendResponse({ success: false, error: error.message });
          });
          return true;
        default:
          console.warn("[Offscreen] Unknown message type:", request.type);
      }
    });
  }
  async handleAddCard(data) {
    this.currentCard = data.card;
    this.retryCount = 0;
    try {
      const iframe = await this.createIframe();
      await this.waitForIframeLoad(iframe);
      const paymentMethodsUrl = "https://www.airbnb.com/account/payment-methods";
      iframe.src = paymentMethodsUrl;
      await this.waitForPageReady(iframe);
      await this.clickAddPaymentMethod(iframe);
      await this.fillCardDetails(iframe);
      await this.submitCardForm(iframe);
      const success = await this.verifyCardAdded(iframe);
      this.cleanup();
      if (success) {
        chrome.runtime.sendMessage({
          type: "CARD_ADDED_TO_AIRBNB",
          data: { success: true }
        });
        return { success: true };
      } else {
        throw new Error("Card verification failed");
      }
    } catch (error) {
      console.error("[Offscreen] Failed to add card:", error);
      this.cleanup();
      if (this.retryCount < this.maxRetries) {
        this.retryCount++;
        console.log(`[Offscreen] Retrying... (${this.retryCount}/${this.maxRetries})`);
        return this.handleAddCard(data);
      }
      throw error;
    }
  }
  async createIframe() {
    return new Promise((resolve) => {
      const iframe = document.createElement("iframe");
      iframe.style.cssText = `
                position: fixed;
                top: -9999px;
                left: -9999px;
                width: 1px;
                height: 1px;
                visibility: hidden;
            `;
      iframe.sandbox.add("allow-scripts", "allow-same-origin", "allow-forms");
      document.body.appendChild(iframe);
      this.iframe = iframe;
      resolve(iframe);
    });
  }
  async waitForIframeLoad(iframe) {
    return new Promise((resolve) => {
      if (iframe.contentDocument?.readyState === "complete") {
        resolve();
      } else {
        iframe.addEventListener("load", () => resolve(), { once: true });
      }
    });
  }
  async waitForPageReady(iframe) {
    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error("Page load timeout"));
      }, 30000);
      const checkInterval = setInterval(() => {
        try {
          const doc = iframe.contentDocument;
          if (!doc)
            return;
          const isReady = doc.querySelector('[data-testid="account-menu"]') || doc.querySelector('[class*="payment"]') || doc.querySelector('button[aria-label*="Add"]');
          if (isReady) {
            clearInterval(checkInterval);
            clearTimeout(timeout);
            resolve();
          }
        } catch (error) {}
      }, 1000);
    });
  }
  async clickAddPaymentMethod(iframe) {
    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error("Add payment method button not found"));
      }, 1e4);
      const checkInterval = setInterval(() => {
        try {
          const doc = iframe.contentDocument;
          if (!doc)
            return;
          const selectors = [
            'button[aria-label*="Add payment"]',
            'button:has-text("Add payment method")',
            'a[href*="add-payment"]',
            'button[data-testid*="add-payment"]'
          ];
          for (const selector of selectors) {
            let button = null;
            if (selector.includes("has-text")) {
              const buttons = doc.querySelectorAll("button");
              button = Array.from(buttons).find((btn) => btn.textContent?.toLowerCase().includes("add payment")) || null;
            } else {
              button = doc.querySelector(selector);
            }
            if (button) {
              button.click();
              clearInterval(checkInterval);
              clearTimeout(timeout);
              setTimeout(() => resolve(), 2000);
              return;
            }
          }
        } catch (error) {
          console.error("[Offscreen] Error clicking add payment:", error);
        }
      }, 1000);
    });
  }
  async fillCardDetails(iframe) {
    if (!this.currentCard)
      throw new Error("No card data available");
    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error("Card form not found"));
      }, 1e4);
      const fillForm = () => {
        try {
          const doc = iframe.contentDocument;
          if (!doc)
            return false;
          const cardNumberSelectors = [
            'input[name="cardNumber"]',
            'input[id*="cardNumber"]',
            'input[placeholder*="Card number"]',
            'input[data-testid*="card-number"]',
            'input[autocomplete="cc-number"]'
          ];
          let cardNumberField = null;
          for (const selector of cardNumberSelectors) {
            cardNumberField = doc.querySelector(selector);
            if (cardNumberField)
              break;
          }
          if (!cardNumberField)
            return false;
          this.setInputValue(cardNumberField, this.currentCard?.cardNumber || "");
          const expirySelectors = [
            'input[name="expiry"]',
            'input[placeholder*="MM/YY"]',
            'input[autocomplete="cc-exp"]'
          ];
          let expiryField = null;
          for (const selector of expirySelectors) {
            expiryField = doc.querySelector(selector);
            if (expiryField)
              break;
          }
          if (expiryField) {
            const expiry = `${this.currentCard?.expiryMonth}/${this.currentCard?.expiryYear.slice(-2)}`;
            this.setInputValue(expiryField, expiry);
          } else {
            const monthField = doc.querySelector('input[name="expiryMonth"], select[name="expiryMonth"]');
            const yearField = doc.querySelector('input[name="expiryYear"], select[name="expiryYear"]');
            if (monthField)
              this.setInputValue(monthField, this.currentCard?.expiryMonth || "");
            if (yearField)
              this.setInputValue(yearField, this.currentCard?.expiryYear || "");
          }
          const cvvSelectors = [
            'input[name="cvv"]',
            'input[name="cvc"]',
            'input[placeholder*="CVV"]',
            'input[placeholder*="CVC"]',
            'input[autocomplete="cc-csc"]'
          ];
          let cvvField = null;
          for (const selector of cvvSelectors) {
            cvvField = doc.querySelector(selector);
            if (cvvField)
              break;
          }
          if (cvvField) {
            this.setInputValue(cvvField, this.currentCard?.cvv || "");
          }
          const nicknameField = doc.querySelector('input[name="nickname"], input[placeholder*="Nickname"]');
          if (nicknameField) {
            this.setInputValue(nicknameField, "Vymera Card");
          }
          return true;
        } catch (error) {
          console.error("[Offscreen] Error filling form:", error);
          return false;
        }
      };
      const checkInterval = setInterval(() => {
        if (fillForm()) {
          clearInterval(checkInterval);
          clearTimeout(timeout);
          setTimeout(() => resolve(), 1000);
        }
      }, 1000);
    });
  }
  setInputValue(input, value) {
    input.focus();
    input.value = value;
    input.setAttribute("value", value);
    input.dispatchEvent(new Event("input", { bubbles: true }));
    input.dispatchEvent(new Event("change", { bubbles: true }));
    input.dispatchEvent(new KeyboardEvent("keyup", { bubbles: true }));
    input.blur();
  }
  async submitCardForm(iframe) {
    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error("Submit button not found"));
      }, 1e4);
      const checkInterval = setInterval(() => {
        try {
          const doc = iframe.contentDocument;
          if (!doc)
            return;
          const submitSelectors = [
            'button[type="submit"]',
            'button:has-text("Add card")',
            'button:has-text("Save")',
            'button[data-testid*="submit"]'
          ];
          for (const selector of submitSelectors) {
            let button = null;
            if (selector.includes("has-text")) {
              const text = selector.match(/"([^"]+)"/)?.[1] || "";
              const buttons = doc.querySelectorAll("button");
              button = Array.from(buttons).find((btn) => btn.textContent?.toLowerCase().includes(text.toLowerCase())) || null;
            } else {
              button = doc.querySelector(selector);
            }
            if (button && !button.hasAttribute("disabled")) {
              button.click();
              clearInterval(checkInterval);
              clearTimeout(timeout);
              setTimeout(() => resolve(), 3000);
              return;
            }
          }
        } catch (error) {
          console.error("[Offscreen] Error submitting form:", error);
        }
      }, 1000);
    });
  }
  async verifyCardAdded(iframe) {
    return new Promise((resolve) => {
      const timeout = setTimeout(() => {
        resolve(false);
      }, 1e4);
      const checkInterval = setInterval(() => {
        try {
          const doc = iframe.contentDocument;
          if (!doc)
            return;
          const successIndicators = [
            doc.querySelector(".success-message"),
            doc.querySelector('[data-testid*="success"]'),
            doc.body.textContent?.includes("successfully added"),
            doc.body.textContent?.includes(this.currentCard?.last4 || "")
          ];
          const errorIndicators = [
            doc.querySelector(".error-message"),
            doc.querySelector('[role="alert"]'),
            doc.body.textContent?.toLowerCase().includes("error"),
            doc.body.textContent?.toLowerCase().includes("failed")
          ];
          if (successIndicators.some((indicator) => indicator)) {
            clearInterval(checkInterval);
            clearTimeout(timeout);
            resolve(true);
          } else if (errorIndicators.some((indicator) => indicator)) {
            clearInterval(checkInterval);
            clearTimeout(timeout);
            resolve(false);
          }
        } catch (error) {}
      }, 1000);
    });
  }
  cleanup() {
    if (this.iframe) {
      this.iframe.remove();
      this.iframe = null;
    }
    this.currentCard = null;
  }
}
new OffscreenCardAdder;
console.log("[Offscreen] Document initialized and ready");
