// extension/content/content.js
class VymeraInjector {
  observer = null;
  buttonInjected = false;
  checkoutDetected = false;
  constructor() {
    this.init();
  }
  init() {
    console.log("[Vymera] Extension initialized on:", window.location.href);
    console.log("[Vymera] Current URL pathname:", window.location.pathname);
    console.log("[Vymera] Is booking URL:", window.location.pathname.includes("/book/"));
    this.injectWalletConnector();
    this.showTestIndicator();
    this.observePageChanges();
    this.checkForCheckoutPage();
    if (window.location.pathname.includes("/book/")) {
      console.log("[Vymera] On booking page, trying to inject toggle immediately...");
      setTimeout(() => {
        this.injectCryptoToggle();
      }, 1000);
      setTimeout(() => {
        this.injectCryptoToggle();
      }, 3000);
    }
    setInterval(() => {
      if (!this.buttonInjected) {
        console.log("[Vymera] Periodic check - button not injected yet");
        this.checkForCheckoutPage();
      }
    }, 2000);
  }
  injectWalletConnector() {
    console.log("[Vymera] Injecting wallet connector script...");
    const script = document.createElement("script");
    script.src = chrome.runtime.getURL("inject/wallet-connector.js");
    script.onload = () => {
      console.log("[Vymera] Wallet connector script injected successfully");
    };
    (document.head || document.documentElement).appendChild(script);
  }
  showTestIndicator() {
    const testElement = document.createElement("div");
    testElement.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: rgba(240, 185, 11, 0.1);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            border: 1px solid rgba(41, 117, 97, 0.2);
            color:rgb(71, 71, 71);
            padding: 16px 24px;
            border-radius: 16px;
            z-index: 10000;
            font-weight: 600;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.2);
            animation: slideIn 0.3s ease-out;
        `;
    testElement.textContent = "Vymera Extension Loaded!";
    document.body.appendChild(testElement);
    const style = document.createElement("style");
    style.textContent = `
            @keyframes slideIn {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
    document.head.appendChild(style);
    setTimeout(() => {
      testElement.style.animation = "slideIn 0.3s ease-out reverse";
      setTimeout(() => {
        testElement.remove();
        style.remove();
      }, 300);
    }, 3000);
  }
  observePageChanges() {
    this.observer = new MutationObserver(() => {
      if (!this.checkoutDetected) {
        this.checkForCheckoutPage();
      }
    });
    this.observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  checkForCheckoutPage() {
    const isBookingUrl = window.location.pathname.includes("/book/");
    console.log("[Vymera] Checking for checkout page...");
    console.log("[Vymera] Is booking URL:", isBookingUrl);
    console.log("[Vymera] Current URL:", window.location.href);
    if (isBookingUrl) {
      console.log("[Vymera] Detected booking URL, looking for checkout elements...");
    }
    const checkoutSelectors = [
      '[data-testid="book-it-default"]',
      '[data-testid="reservation-payment"]',
      '[data-testid="checkout-page"]',
      'button[type="submit"][data-testid*="reservation"]',
      'div[aria-label*="payment"]',
      'div[class*="payment-section"]',
      'div[class*="checkout"]',
      'h2:has-text("Confirm and pay")',
      'h1:has-text("Confirm and pay")',
      'button:contains("Confirm and pay")',
      'h1:contains("Confirm and pay")',
      'h2:contains("Confirm and pay")'
    ];
    let isCheckout = isBookingUrl;
    if (!isCheckout) {
      isCheckout = checkoutSelectors.some((selector) => {
        try {
          if (selector.includes(":contains")) {
            const match = selector.match(/(.+):contains\("(.+)"\)/);
            if (match) {
              const [, elementSelector, text] = match;
              const elements = document.querySelectorAll(elementSelector);
              return Array.from(elements).some((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
            }
          } else if (selector.includes("has-text")) {
            const element = selector.split(":")[0];
            const text = selector.match(/"([^"]+)"/)?.[1];
            if (text) {
              const elements = document.querySelectorAll(element);
              return Array.from(elements).some((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
            }
          } else {
            return document.querySelector(selector) !== null;
          }
        } catch {
          return false;
        }
        return false;
      });
    }
    if (isCheckout && !this.buttonInjected) {
      console.log("[Vymera] Checkout detected, injecting crypto toggle...");
      this.checkoutDetected = true;
      this.injectCryptoToggle();
      this.buttonInjected = true;
    }
  }
  injectCryptoToggle() {
    console.log("[Vymera] Creating custom crypto toggle...");
    if (document.querySelector(".vymera-crypto-toggle") || document.querySelector('[data-vymera-crypto-toggle="true"]')) {
      console.log("[Vymera] Crypto toggle already exists");
      return;
    }
    const workTripSelectors = [
      'div[data-plugin-in-point-id="SWITCH_ROW_WORK_TRIP"]',
      'div:contains("Is this a work trip?")',
      'div:has-text("Is this a work trip?")',
      'div[class*="work-trip"]',
      'div[class*="work_trip"]'
    ];
    let workTripToggle = null;
    for (const selector of workTripSelectors) {
      try {
        if (selector.includes(":contains")) {
          const match = selector.match(/(.+):contains\("(.+)"\)/);
          if (match) {
            const [, elementSelector, text] = match;
            const elements = document.querySelectorAll(elementSelector);
            workTripToggle = Array.from(elements).find((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
          }
        } else if (selector.includes("has-text")) {
          const element = selector.split(":")[0];
          const text = selector.match(/"([^"]+)"/)?.[1];
          if (text) {
            const elements = document.querySelectorAll(element);
            workTripToggle = Array.from(elements).find((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
          }
        } else {
          workTripToggle = document.querySelector(selector);
        }
        if (workTripToggle) {
          console.log("[Vymera] Found work trip toggle with selector:", selector);
          break;
        }
      } catch (e) {
        console.debug("[Vymera] Error with selector:", selector, e);
      }
    }
    if (!workTripToggle) {
      console.log("[Vymera] Work trip toggle not found with any selector");
      console.log("[Vymera] Available elements with data-plugin-in-point-id:", document.querySelectorAll("[data-plugin-in-point-id]"));
      console.log('[Vymera] Available elements containing "work trip":', document.querySelectorAll("*").length);
      const allElements = document.querySelectorAll("*");
      for (const el of allElements) {
        if (el.textContent?.toLowerCase().includes("work trip")) {
          console.log('[Vymera] Found element with "work trip" text:', el);
          workTripToggle = el;
          break;
        }
      }
    }
    if (!workTripToggle) {
      console.log("[Vymera] Still no work trip toggle found, trying to inject at a different location...");
      const anyToggle = document.querySelector('[role="switch"], [data-testid*="switch"], [class*="switch"]');
      if (anyToggle) {
        console.log("[Vymera] Found alternative toggle element:", anyToggle);
        workTripToggle = anyToggle;
      } else {
        console.log("[Vymera] No suitable injection point found, trying fallback injection...");
        this.injectCryptoToggleFallback();
        return;
      }
    }
    console.log("[Vymera] Found work trip toggle, creating custom crypto toggle...", workTripToggle);
    const cryptoToggle = document.createElement("div");
    cryptoToggle.className = "vymera-crypto-toggle";
    cryptoToggle.setAttribute("data-vymera-crypto-toggle", "true");
    cryptoToggle.setAttribute("data-plugin-in-point-id", "SWITCH_ROW_CRYPTO");
    cryptoToggle.setAttribute("data-section-id", "SWITCH_ROW_CRYPTO");
    cryptoToggle.style.cssText = `
            padding-top: 32px;
            padding-bottom: 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            border-top-style: solid;
            border-top-color: lightgray;
            border-top-width: 1px;
        `;
    const titleDiv = document.createElement("div");
    titleDiv.className = "t41l3z9 atm_c8_9oan3l atm_g3_1dzntr8 atm_cs_18jqzaw dir dir-ltr";
    titleDiv.id = "SWITCH_ROW_CRYPTO-title";
    titleDiv.textContent = "Pay with crypto";
    titleDiv.style.cssText = `
            font-weight: 500;
            line-height: 1.5;
            font-size: 16px;
            color: fff;
        `;
    const toggleButton = document.createElement("button");
    toggleButton.className = "vymera-crypto-switch";
    toggleButton.setAttribute("role", "switch");
    toggleButton.setAttribute("aria-checked", "false");
    toggleButton.setAttribute("aria-labelledby", "SWITCH_ROW_CRYPTO-title");
    toggleButton.id = "SWITCH_ROW_CRYPTO-switch";
    toggleButton.style.cssText = `
            width: 48px;
            height: 32px;
            background: #e0e0e0;
            border: none;
            border-radius: 48px;
            position: relative;
            cursor: pointer;
            transition: background-color 0.3s ease;
            outline: none;
        `;
    const toggleKnob = document.createElement("div");
    toggleKnob.style.cssText = `
            width: 28px;
            height: 28px;
            background: white;
            border-radius: 50%;
            position: absolute;
            top: 2px;
            left: 2px;
            transition: transform 0.3s ease;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        `;
    toggleButton.appendChild(toggleKnob);
    cryptoToggle.appendChild(titleDiv);
    cryptoToggle.appendChild(toggleButton);
    workTripToggle.parentElement?.insertBefore(cryptoToggle, workTripToggle.nextSibling);
    this.setupToggleFunctionality(cryptoToggle);
    console.log("[Vymera] Custom crypto toggle added successfully");
    this.showSuccessIndicator();
  }
  injectCryptoToggleFallback() {
    console.log("[Vymera] Using fallback injection method...");
    const possibleInjectionPoints = [
      'div[class*="payment"]',
      'div[class*="checkout"]',
      'div[class*="booking"]',
      'div[aria-label*="payment"]',
      'div[data-testid*="payment"]',
      'div[data-testid*="checkout"]',
      'div[data-testid*="booking"]'
    ];
    let injectionPoint = null;
    for (const selector of possibleInjectionPoints) {
      const element = document.querySelector(selector);
      if (element) {
        console.log("[Vymera] Found injection point with selector:", selector);
        injectionPoint = element;
        break;
      }
    }
    if (!injectionPoint) {
      console.log("[Vymera] No injection point found, injecting at body");
      injectionPoint = document.body;
    }
    const cryptoToggle = document.createElement("div");
    cryptoToggle.className = "vymera-crypto-toggle";
    cryptoToggle.setAttribute("data-vymera-crypto-toggle", "true");
    cryptoToggle.style.cssText = `
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 16px;
            margin: 16px 0;
            position: relative;
            z-index: 1000;
        `;
    cryptoToggle.innerHTML = `
            <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 12px;">
                <img src="https://bscscan.com/assets/bsc/images/svg/logos/token-light.svg?v=25.9.4.0" alt="BNB Token" style="width: 24px; height: 24px; border-radius: 4px; object-fit: contain;">
                <div>
                    <h3 style="margin: 0; font-size: 16px; font-weight: 600; color: #fff;">Pay with crypto</h3>
                    <p style="margin: 4px 0 0 0; font-size: 12px; color: #666;">Secure BNB payment</p>
                </div>
            </div>
            <button id="vymera-crypto-toggle-button" style="
                width: 100%;
                background: #297561;
                color: #000;
                border: none;
                border-radius: 6px;
                padding: 12px;
                font-size: 14px;
                font-weight: 600;
                cursor: pointer;
                transition: background-color 0.2s ease;
            ">Connect Wallet</button>
        `;
    if (injectionPoint === document.body) {
      injectionPoint.appendChild(cryptoToggle);
    } else {
      injectionPoint.insertBefore(cryptoToggle, injectionPoint.firstChild);
    }
    const toggleButton = cryptoToggle.querySelector("#vymera-crypto-toggle-button");
    toggleButton.addEventListener("click", async () => {
      try {
        await this.connectWallet();
        console.log("[Vymera] Wallet connected via fallback method");
        toggleButton.textContent = "Wallet Connected ✓";
        toggleButton.style.background = "#4CAF50";
        toggleButton.style.color = "white";
      } catch (error) {
        console.error("[Vymera] Wallet connection failed:", error);
        toggleButton.textContent = "Connection Failed";
        toggleButton.style.background = "#f44336";
        toggleButton.style.color = "white";
        setTimeout(() => {
          toggleButton.textContent = "Connect Wallet";
          toggleButton.style.background = "#297561";
          toggleButton.style.color = "#000";
        }, 3000);
      }
    });
    console.log("[Vymera] Fallback crypto toggle added successfully");
    this.showSuccessIndicator();
  }
  setupToggleFunctionality(cryptoToggle) {
    const toggleButton = cryptoToggle.querySelector(".vymera-crypto-switch");
    const toggleKnob = cryptoToggle.querySelector(".vymera-crypto-switch > div");
    let isCryptoEnabled = false;
    let originalPaymentSection = null;
    const updateToggle = async (enabled) => {
      isCryptoEnabled = enabled;
      if (enabled) {
        toggleButton.style.background = "#297561";
        toggleKnob.style.transform = "translateX(16px)";
        toggleButton.setAttribute("aria-checked", "true");
        if (!toggleKnob.querySelector(".tick-mark")) {
          const tickMark = document.createElement("div");
          tickMark.className = "tick-mark";
          tickMark.style.cssText = `
                        position: absolute;
                        top: 50%;
                        left: 50%;
                        transform: translate(-50%, -50%);
                        color: #297561;
                        font-size: 16px;
                        font-weight: bold;
                        line-height: 1;
                    `;
          tickMark.innerHTML = '<svg viewBox="0 0 12 12" xmlns="http://www.w3.org/2000/svg" aria-hidden="true" role="presentation" focusable="false" style="display: block; height: 12px; width: 12px; fill: currentcolor;"><path d="m10.5 1.939 1.061 1.061-7.061 7.061-.53-.531-3-3-.531-.53 1.061-1.061 3 3 5.47-5.469z"></path></svg>';
          toggleKnob.appendChild(tickMark);
        }
        this.replaceConfirmPayButton();
        this.hideQuickPayTerms();
        this.ensureOriginalButtonHidden();
      } else {
        toggleButton.style.background = "#e0e0e0";
        toggleKnob.style.transform = "translateX(0)";
        toggleButton.setAttribute("aria-checked", "false");
        const tickMark = toggleKnob.querySelector(".tick-mark");
        if (tickMark) {
          tickMark.remove();
        }
        this.hideCryptoPaymentOptions(originalPaymentSection);
        this.showQuickPayTerms();
      }
      if (!originalPaymentSection) {
        originalPaymentSection = this.findPaymentSection();
      }
      if (enabled && originalPaymentSection) {
        if (!originalPaymentSection.dataset.originalContent) {
          originalPaymentSection.dataset.originalContent = originalPaymentSection.innerHTML;
        }
        await this.showCryptoPaymentOptions(originalPaymentSection);
      } else if (originalPaymentSection) {
        this.hideCryptoPaymentOptions(originalPaymentSection);
      }
    };
    toggleButton.addEventListener("click", (e) => {
      e.preventDefault();
      e.stopPropagation();
      updateToggle(!isCryptoEnabled);
    });
  }
  findPaymentSection() {
    const paymentSectionSelectors = [
      'div[aria-label*="Pay with"]',
      'div[class*="payment"]',
      'section:has(h2:contains("Pay with"))',
      'div:has(> div > h2:contains("Pay with"))',
      'div:has(button:contains("Mastercard"))',
      'div:has(select option[value*="Mastercard"])'
    ];
    for (const selector of paymentSectionSelectors) {
      try {
        if (selector.includes(":contains")) {
          const match = selector.match(/(.+):contains\("(.+)"\)/);
          if (match) {
            const [, elementSelector, text] = match;
            const elements = document.querySelectorAll(elementSelector);
            const found = Array.from(elements).find((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
            if (found) {
              return found;
            }
          }
        } else {
          const element = document.querySelector(selector);
          if (element) {
            return element;
          }
        }
      } catch (e) {
        console.debug("[Vymera] Error with selector:", selector, e);
      }
    }
    return null;
  }
  showSuccessIndicator() {
    const successElement = document.createElement("div");
    successElement.style.cssText = `
            position: fixed;
            top: 80px;
            right: 20px;
            background: rgba(76, 175, 80, 0.1);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            border: 1px solid rgba(76, 175, 80, 0.2);
            color: #4CAF50;
            padding: 16px 24px;
            border-radius: 16px;
            z-index: 10000;
            font-weight: 600;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.2);
            animation: slideIn 0.3s ease-out;
        `;
    successElement.textContent = "Crypto Toggle Added!";
    document.body.appendChild(successElement);
    setTimeout(() => {
      successElement.style.animation = "slideIn 0.3s ease-out reverse";
      setTimeout(() => {
        successElement.remove();
      }, 300);
    }, 3000);
  }
  hideQuickPayTerms() {
    console.log("[Vymera] Hiding quick-pay terms and conditions...");
    const termsSelectors = [
      "#quick-pay-terms-and-conditions",
      '[id="quick-pay-terms-and-conditions"]',
      '[class*="quick-pay-terms-and-conditions"]',
      '[data-testid*="quick-pay-terms"]',
      '[class*="terms-and-conditions"]',
      'p:contains("terms and conditions")',
      'div:contains("terms and conditions")'
    ];
    for (const selector of termsSelectors) {
      try {
        if (selector.includes(":contains")) {
          const match = selector.match(/(.+):contains\("(.+)"\)/);
          if (match) {
            const [, elementSelector, text] = match;
            const elements = document.querySelectorAll(elementSelector);
            const found = Array.from(elements).find((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
            if (found) {
              const element = found;
              element.style.display = "none";
              element.setAttribute("data-vymera-hidden", "true");
              console.log("[Vymera] Hidden quick-pay terms:", element);
            }
          }
        } else {
          const element = document.querySelector(selector);
          if (element) {
            element.style.display = "none";
            element.setAttribute("data-vymera-hidden", "true");
            console.log("[Vymera] Hidden quick-pay terms:", element);
          }
        }
      } catch (e) {
        console.debug("[Vymera] Error with terms selector:", selector, e);
      }
    }
  }
  showQuickPayTerms() {
    console.log("[Vymera] Showing quick-pay terms and conditions...");
    const hiddenElements = document.querySelectorAll('[data-vymera-hidden="true"]');
    hiddenElements.forEach((element) => {
      const htmlElement = element;
      htmlElement.style.display = "";
      htmlElement.removeAttribute("data-vymera-hidden");
      console.log("[Vymera] Shown quick-pay terms:", htmlElement);
    });
  }
  ensureOriginalButtonHidden() {
    if (this.originalConfirmPayButton) {
      console.log("[Vymera] Ensuring original button stays hidden...");
      this.originalConfirmPayButton.style.display = "none";
      this.originalConfirmPayButton.setAttribute("data-vymera-hidden", "true");
    }
  }
  async checkWalletConnection() {
    try {
      console.log("[Vymera] Checking wallet connection via injected script...");
      const result = await new Promise((resolve) => {
        const handleResponse = (event) => {
          window.removeEventListener("vymera-wallet-response", handleResponse);
          resolve({
            success: event.detail.success,
            account: event.detail.account
          });
        };
        window.addEventListener("vymera-wallet-response", handleResponse);
        window.dispatchEvent(new CustomEvent("vymera-connect-wallet"));
        setTimeout(() => {
          window.removeEventListener("vymera-wallet-response", handleResponse);
          resolve({ success: false, account: null });
        }, 5000);
      });
      if (result.success && result.account) {
        this.connectedAccount = result.account;
        console.log("[Vymera] Connected account set:", result.account);
      }
      console.log("[Vymera] Wallet connection status:", result.success);
      return result.success;
    } catch (error) {
      console.log("[Vymera] Wallet connection check failed:", error);
      return false;
    }
  }
  async waitForEthereum(timeout = 3000) {
    console.log("[Vymera] Checking for ethereum provider via injected script...");
    return new Promise((resolve) => {
      const handleResponse = (event) => {
        window.removeEventListener("vymera-wallet-response", handleResponse);
        if (event.detail.success) {
          console.log("[Vymera] Ethereum provider available via injected script");
          resolve(true);
        } else {
          console.log("[Vymera] Ethereum provider not available");
          resolve(false);
        }
      };
      window.addEventListener("vymera-wallet-response", handleResponse);
      window.dispatchEvent(new CustomEvent("vymera-connect-wallet"));
      setTimeout(() => {
        window.removeEventListener("vymera-wallet-response", handleResponse);
        console.log("[Vymera] Timeout waiting for ethereum provider");
        resolve(false);
      }, timeout);
    });
  }
  connectedAccount = null;
  async connectWallet() {
    console.log("[Vymera] Attempting to connect wallet via custom events...");
    return new Promise((resolve, reject) => {
      const handleResponse = (event) => {
        console.log("[Vymera] Received wallet response:", event.detail);
        window.removeEventListener("vymera-wallet-response", handleResponse);
        if (event.detail.success) {
          console.log("[Vymera] Wallet connected successfully!");
          console.log("[Vymera] Account:", event.detail.account);
          console.log("[Vymera] Chain ID:", event.detail.chainId);
          this.connectedAccount = event.detail.account;
          if (event.detail.chainId !== "0x38") {
            this.switchToBSCNetwork().then(() => resolve()).catch(reject);
          } else {
            resolve();
          }
        } else {
          console.error("[Vymera] Wallet connection failed:", event.detail.error);
          reject(new Error(event.detail.error));
        }
      };
      window.addEventListener("vymera-wallet-response", handleResponse);
      console.log("[Vymera] Dispatching wallet connection request...");
      window.dispatchEvent(new CustomEvent("vymera-connect-wallet"));
      setTimeout(() => {
        window.removeEventListener("vymera-wallet-response", handleResponse);
        reject(new Error("Wallet connection timeout"));
      }, 30000);
    });
  }
  async switchToBSCNetwork() {
    console.log("[Vymera] Attempting to switch to BSC network...");
    return new Promise((resolve, reject) => {
      const handleResponse = (event) => {
        console.log("[Vymera] Received network switch response:", event.detail);
        window.removeEventListener("vymera-network-response", handleResponse);
        if (event.detail.success) {
          console.log("[Vymera] Network switch successful:", event.detail.message);
          resolve();
        } else {
          console.error("[Vymera] Network switch failed:", event.detail.error);
          resolve();
        }
      };
      window.addEventListener("vymera-network-response", handleResponse);
      console.log("[Vymera] Dispatching network switch request...");
      window.dispatchEvent(new CustomEvent("vymera-switch-network"));
      setTimeout(() => {
        window.removeEventListener("vymera-network-response", handleResponse);
        console.warn("[Vymera] Network switch timeout - continuing anyway");
        resolve();
      }, 1e4);
    });
  }
  async switchToBlockchain(chainId, chainName) {
    console.log(`[Vymera] Attempting to switch to ${chainName} (${chainId})...`);
    return new Promise((resolve, reject) => {
      const handleResponse = (event) => {
        console.log("[Vymera] Received blockchain switch response:", event.detail);
        window.removeEventListener("vymera-blockchain-response", handleResponse);
        if (event.detail.success) {
          console.log("[Vymera] Blockchain switch successful:", event.detail.message);
          this.showNetworkSwitchNotification(chainName);
          resolve();
        } else {
          console.error("[Vymera] Blockchain switch failed:", event.detail.error);
          reject(new Error(event.detail.error));
        }
      };
      window.addEventListener("vymera-blockchain-response", handleResponse);
      console.log("[Vymera] Dispatching blockchain switch request...");
      window.dispatchEvent(new CustomEvent("vymera-switch-blockchain", {
        detail: { chainId, chainName }
      }));
      setTimeout(() => {
        window.removeEventListener("vymera-blockchain-response", handleResponse);
        console.warn("[Vymera] Blockchain switch timeout - continuing anyway");
        resolve();
      }, 1e4);
    });
  }
  showNetworkSwitchNotification(chainName) {
    const notification = document.createElement("div");
    notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: rgba(33, 150, 243, 0.1);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid rgba(33, 150, 243, 0.3);
            color: #333;
            padding: 20px 28px;
            border-radius: 20px;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.37);
            z-index: 10000;
            font-weight: 600;
            animation: slideIn 0.3s ease-out;
            width: 300px;
            max-width: 300px;
        `;
    notification.innerHTML = `
            <div style="display: flex; align-items: center; gap: 12px;">
                <div style="
                    width: 24px;
                    height: 24px;
                    background: #2196F3;
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    font-size: 14px;
                    font-weight: bold;
                ">✓</div>
                <div>
                    <div style="font-size: 16px; font-weight: 600; margin-bottom: 4px;">Network Switched</div>
                    <div style="font-size: 14px; color: #666;">Connected to ${chainName}</div>
                </div>
            </div>
        `;
    document.body.appendChild(notification);
    if (!document.querySelector("#network-switch-animation")) {
      const style = document.createElement("style");
      style.id = "network-switch-animation";
      style.textContent = `
                @keyframes slideIn {
                    from {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
            `;
      document.head.appendChild(style);
    }
    setTimeout(() => {
      notification.style.animation = "slideIn 0.3s ease-out reverse";
      setTimeout(() => {
        notification.remove();
      }, 300);
    }, 3000);
  }
  async fetchTokenBalances() {
    console.log("[Vymera] Fetching token balances...");
    return new Promise((resolve, reject) => {
      const handleResponse = (event) => {
        console.log("[Vymera] Received balances response:", event.detail);
        window.removeEventListener("vymera-balances-response", handleResponse);
        if (event.detail.success) {
          resolve(event.detail.balances);
        } else {
          console.error("[Vymera] Balance fetch failed:", event.detail.error);
          resolve({ BNB: "0.0000", ETH: "0.0000", USDT: "0.0000", USDC: "0.0000" });
        }
      };
      window.addEventListener("vymera-balances-response", handleResponse);
      console.log("[Vymera] Dispatching balance fetch request...");
      window.dispatchEvent(new CustomEvent("vymera-fetch-balances"));
      setTimeout(() => {
        window.removeEventListener("vymera-balances-response", handleResponse);
        resolve({ BNB: "0.0000", ETH: "0.0000", USDT: "0.0000", USDC: "0.0000" });
      }, 5000);
    });
  }
  originalConfirmPayButton = null;
  cryptoPayButton = null;
  payButtonReplaced = false;
  replaceConfirmPayButton() {
    if (this.payButtonReplaced) {
      return;
    }
    const confirmPaySelectors = [
      'button[type="submit"]:contains("Confirm and pay")',
      'button:contains("Confirm and pay")',
      'button[aria-label*="Confirm and pay"]',
      'button[data-testid*="confirm"]',
      'button[type="submit"][class*="primary"]',
      'button[type="submit"]'
    ];
    let confirmButton = null;
    for (const selector of confirmPaySelectors) {
      try {
        if (selector.includes(":contains")) {
          const match = selector.match(/(.+):contains\("(.+)"\)/);
          if (match) {
            const [, elementSelector, text] = match;
            const elements = document.querySelectorAll(elementSelector);
            confirmButton = Array.from(elements).find((el) => el.textContent?.toLowerCase().includes(text.toLowerCase()));
          }
        } else {
          const elements = document.querySelectorAll(selector);
          confirmButton = Array.from(elements).find((el) => el.textContent?.toLowerCase().includes("confirm and pay"));
          if (!confirmButton && elements.length > 0) {
            confirmButton = elements[0];
          }
        }
        if (confirmButton) {
          console.log("[Vymera] Found confirm button with selector:", selector);
          break;
        }
      } catch (e) {
        console.debug("[Vymera] Error with selector:", selector, e);
      }
    }
    if (!confirmButton) {
      console.log("[Vymera] Confirm and pay button not found yet");
      return;
    }
    console.log("[Vymera] Found Confirm and pay button, replacing...");
    this.originalConfirmPayButton = confirmButton;
    confirmButton.style.display = "none";
    confirmButton.setAttribute("data-vymera-hidden", "true");
    this.payButtonReplaced = true;
    console.log("[Vymera] Confirm and pay button replaced with crypto button");
  }
  restoreConfirmPayButton() {
    if (!this.payButtonReplaced || !this.originalConfirmPayButton || !this.cryptoPayButton) {
      return;
    }
    console.log("[Vymera] Restoring original confirm and pay button...");
    this.originalConfirmPayButton.style.display = "";
    this.cryptoPayButton.remove();
    this.cryptoPayButton = null;
    this.payButtonReplaced = false;
    console.log("[Vymera] Original button restored");
  }
  async handleCryptoPayment() {
    console.log("[Vymera] Starting crypto payment flow...");
    try {
      const isConnected = await this.checkWalletConnection();
      if (!isConnected) {
        console.log("[Vymera] Wallet not connected, connecting...");
        await this.connectWallet();
      }
      const calculation = await this.calculateBNBAmount();
      console.log("[Vymera] Payment calculation:", calculation);
      if (!this.connectedAccount) {
        const accounts = await new Promise((resolve) => {
          window.addEventListener("vymera-wallet-response", (event) => {
            if (event.detail.success) {
              resolve([event.detail.account]);
            } else {
              resolve([]);
            }
          }, { once: true });
          window.dispatchEvent(new CustomEvent("vymera-connect-wallet"));
        });
        if (accounts.length > 0) {
          this.connectedAccount = accounts[0];
        } else {
          throw new Error("No wallet connected");
        }
      }
      const vymeraContract = "0x9cb048e45aAA295Ebb4a9b3dEcb05c529C4C6D88";
      const bnbAsNumber = parseFloat(calculation.bnbAmount);
      const weiAmount = BigInt(Math.floor(bnbAsNumber * Math.pow(10, 18)));
      const paymentId = await this.generateFundingId();
      const fiatAmountInCents = BigInt(Math.floor(parseFloat(calculation.totalPrice.amount) * 100));
      const functionSelector = this.calculateFunctionSelector("fund(uint256,address,uint256,string,uint256)");
      const currencyCode = calculation.totalPrice.currency === "USD" ? "USD" : "SGD";
      const encodedParams = this.encodeFundParams(paymentId, "0x0000000000000000000000000000000000000000", weiAmount, currencyCode, fiatAmountInCents);
      const data = functionSelector + encodedParams;
      let currentChainId, currentNetwork;
      try {
        currentChainId = await window.ethereum.request({ method: "eth_chainId" });
        const networkNames = {
          "0x14a34": "Base Sepolia",
          "0x38": "BNB Smart Chain",
          "0xef": "TAC Mainnet",
          "0xa4ec": "Celo Mainnet",
          "0x504": "Moonbeam"
        };
        currentNetwork = networkNames[currentChainId] || `Chain ${parseInt(currentChainId, 16)}`;
      } catch (error) {
        console.warn("[Vymera] Could not get current chain ID:", error);
        currentChainId = "0x38";
        currentNetwork = "BNB Smart Chain";
      }
      console.log("[Vymera] Calling fund() on contract:", {
        contract: vymeraContract,
        network: currentNetwork,
        chainId: currentChainId,
        id: paymentId.toString(),
        tokenAddress: "0x0000000000000000000000000000000000000000",
        tokenAmount: weiAmount.toString(),
        currencyCode,
        fiatAmount: fiatAmountInCents.toString(),
        value: weiAmount.toString()
      });
      console.log("[Vymera] Transaction payload:", {
        functionSelector,
        encodedParams,
        fullData: data,
        dataLength: data.length,
        hexDataLength: (data.length - 2) / 2 + " bytes"
      });
      console.log("[Vymera] Transaction parameters:", {
        from: this.connectedAccount,
        to: vymeraContract,
        value: "0x" + weiAmount.toString(16) + " (" + weiAmount.toString() + " wei)",
        data,
        gas: "0x30D40 (200000)"
      });
      const txHash = await new Promise((resolve, reject) => {
        const handleResponse = (event) => {
          window.removeEventListener("vymera-transaction-response", handleResponse);
          if (event.detail.success) {
            resolve(event.detail.txHash);
          } else {
            reject(new Error(event.detail.error || "Transaction failed"));
          }
        };
        window.addEventListener("vymera-transaction-response", handleResponse);
        window.dispatchEvent(new CustomEvent("vymera-send-transaction", {
          detail: {
            from: this.connectedAccount,
            to: vymeraContract,
            value: "0x" + weiAmount.toString(16),
            data,
            gas: "0x30D40"
          }
        }));
        setTimeout(() => {
          window.removeEventListener("vymera-transaction-response", handleResponse);
          reject(new Error("Transaction timeout"));
        }, 60000);
      });
      console.log("[Vymera] Transaction sent:", txHash);
      const processingMessage = document.createElement("div");
      processingMessage.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: rgba(33, 150, 243, 0.1);
                backdrop-filter: blur(12px);
                -webkit-backdrop-filter: blur(12px);
                border: 1px solid rgba(33, 150, 243, 0.3);
                color: #333;
                padding: 20px 28px;
                border-radius: 20px;
                box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.37);
                z-index: 10000;
                font-weight: 600;
                animation: slideIn 0.3s ease-out;
                width: 300px;
                max-width: 300px;
            `;
      const explorerUrls = {
        "0x14a34": `https://base-sepolia.blockscout.com/tx/${txHash}`,
        "0x38": `https://bscscan.com/tx/${txHash}`,
        "0xef": `https://explorer.tac.build/tx/${txHash}`,
        "0xa4ec": `https://explorer.celo.org/tx/${txHash}`,
        "0x504": `https://moonbeam.moonscan.io/tx/${txHash}`
      };
      const explorerUrl = explorerUrls[currentChainId] || `https://bscscan.com/tx/${txHash}`;
      const explorerName = currentNetwork.includes("Base") ? "Blockscout" : currentNetwork.includes("BSC") ? "BSCScan" : currentNetwork.includes("TAC") ? "TAC Explorer" : currentNetwork.includes("Celo") ? "Celo Explorer" : currentNetwork.includes("Moonbeam") ? "Moonscan" : "Explorer";
      processingMessage.innerHTML = `
                ⏳ Transaction submitted!<br>
                <small>Network: ${currentNetwork}</small><br>
                <small>Hash: ${txHash.substring(0, 10)}...${txHash.substring(58)}</small><br>
                <a href="${explorerUrl}" target="_blank" style="color: #333; text-decoration: underline;">View on ${explorerName}</a>
            `;
      document.body.appendChild(processingMessage);
      setTimeout(() => {
        console.log("[Vymera] Simulating transaction confirmation...");
        processingMessage.innerHTML = "✓ Transaction confirmed!<br><small>Processing payment...</small>";
        processingMessage.style.background = "rgba(255, 152, 0, 0.1)";
        processingMessage.style.borderColor = "rgba(255, 152, 0, 0.3)";
        setTimeout(() => {
          processingMessage.remove();
          const successMessage = document.createElement("div");
          successMessage.style.cssText = `
                        position: fixed;
                        top: 20px;
                        right: 20px;
                        background: rgba(76, 175, 80, 0.1);
                        backdrop-filter: blur(12px);
                        -webkit-backdrop-filter: blur(12px);
                        border: 1px solid rgba(76, 175, 80, 0.3);
                        color: #333;
                        padding: 20px 28px;
                        border-radius: 20px;
                        box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.37);
                        z-index: 10000;
                        font-weight: 600;
                        animation: slideIn 0.3s ease-out;
                        width: 300px;
                        max-width: 300px;
                    `;
          successMessage.innerHTML = "<span style='font-size: 20px;'>✓</span> Crypto payment successful!<br><small style='opacity: 0.9;'>Completing your booking...</small>";
          document.body.appendChild(successMessage);
          setTimeout(() => {
            if (this.originalConfirmPayButton) {
              console.log("[Vymera] Clicking original Airbnb button...");
              this.originalConfirmPayButton.click();
            }
            successMessage.remove();
          }, 1500);
        }, 2000);
        this.pollForCardDetails(paymentId.toString(), processingMessage);
      }, 3000);
    } catch (error) {
      console.error("[Vymera] Crypto payment failed:", error);
      alert("Payment failed: " + error.message);
    }
  }
  extractPaymentData() {
    const totalPrice = this.extractTotalPrice();
    return {
      amount: totalPrice.amount.toString(),
      currency: totalPrice.currency,
      bookingId: window.location.pathname.split("/").pop() || ""
    };
  }
  encodeFundParams(id, tokenAddress, tokenAmount, currencyCode, fiatAmount) {
    const cleanAddress = tokenAddress.replace(/^0x/, "").toLowerCase();
    const paddedId = id.toString(16).padStart(64, "0");
    const paddedAddress = cleanAddress.padStart(64, "0");
    const paddedTokenAmount = tokenAmount.toString(16).padStart(64, "0");
    const paddedFiatAmount = fiatAmount.toString(16).padStart(64, "0");
    const stringDataOffset = "00000000000000000000000000000000000000000000000000000000000000a0";
    const stringBytes = new TextEncoder().encode(currencyCode);
    const stringLength = stringBytes.length.toString(16).padStart(64, "0");
    let stringHex = "";
    for (let i = 0;i < stringBytes.length; i++) {
      stringHex += stringBytes[i].toString(16).padStart(2, "0");
    }
    const paddingNeeded = 64 - stringHex.length % 64;
    if (paddingNeeded !== 64) {
      stringHex = stringHex.padEnd(stringHex.length + paddingNeeded, "0");
    }
    console.log("[Vymera] Encoding parameters:", {
      id: paddedId,
      tokenAddress: paddedAddress,
      tokenAmount: paddedTokenAmount,
      stringOffset: stringDataOffset,
      fiatAmount: paddedFiatAmount,
      stringLength,
      stringHex,
      currencyCode,
      fullEncoding: paddedId + paddedAddress + paddedTokenAmount + stringDataOffset + paddedFiatAmount + stringLength + stringHex
    });
    return paddedId + paddedAddress + paddedTokenAmount + stringDataOffset + paddedFiatAmount + stringLength + stringHex;
  }
  calculateFunctionSelector(signature) {
    return "0x0cca551c";
  }
  testEncoding() {
    const testId = BigInt("7278606436507629840");
    const testAddress = "0x0000000000000000000000000000000000000000";
    const testAmount = BigInt("78061287565840384");
    const testCurrency = "SGD";
    const testFiat = BigInt("10001");
    const encoded = this.encodeFundParams(testId, testAddress, testAmount, testCurrency, testFiat);
    console.log("[Vymera] Test encoding result:", encoded);
    console.log("[Vymera] Expected result should match your data");
  }
  async showCryptoPaymentOptions(paymentSection) {
    if (!paymentSection.dataset.originalContent) {
      paymentSection.dataset.originalContent = paymentSection.innerHTML;
    }
    const cryptoSection = document.createElement("div");
    cryptoSection.className = "vymera-crypto-payment";
    cryptoSection.style.cssText = `
            background: white;
            border-radius: 8px;
        `;
    const isConnected = await this.checkWalletConnection();
    if (isConnected) {
      this.createConnectedWalletUI(cryptoSection);
    } else {
      this.createWalletConnectionPrompt(cryptoSection);
    }
    paymentSection.innerHTML = "";
    paymentSection.appendChild(cryptoSection);
    console.log("[Vymera] Crypto payment options shown");
  }
  createWalletConnectionPrompt(container) {
    const promptSection = document.createElement("div");
    promptSection.style.cssText = `
            text-align: center;
            padding: 32px 16px;
            border-style: solid;
            border-width: 1px;
            border-color: lightgray;
            border-radius: 16px;
        `;
    const metamaskIcon = document.createElement("div");
    metamaskIcon.style.cssText = `
            width: 64px;
            height: 64px;
            background: #F6851B;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 16px;
            font-size: 32px;
        `;
    metamaskIcon.innerHTML = '<img src="https://images.ctfassets.net/clixtyxoaeas/1ezuBGezqfIeifWdVtwU4c/d970d4cdf13b163efddddd5709164d2e/MetaMask-icon-Fox.svg" style="height: 32px;">';
    const title = document.createElement("h3");
    title.textContent = "Connect Your Wallet";
    title.style.cssText = `
            margin: 0 0 8px 0;
            font-size: 20px;
            font-weight: 600;
            color: #222;
        `;
    const description = document.createElement("p");
    description.textContent = "Connect your MetaMask wallet to pay with cryptocurrency";
    description.style.cssText = `
            margin: 0 0 24px 0;
            color: #666;
            font-size: 14px;
        `;
    const connectButton = document.createElement("button");
    connectButton.textContent = "Connect MetaMask";
    connectButton.style.cssText = `
            background: #F6851B;
            color: white;
            border: none;
            border-radius: 48px;
            padding: 12px 24px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.2s ease;
        `;
    connectButton.addEventListener("click", async (e) => {
      e.preventDefault();
      e.stopPropagation();
      console.log("[Vymera] Connect button clicked!");
      connectButton.disabled = true;
      connectButton.textContent = "Connecting...";
      connectButton.style.background = "#ccc";
      try {
        console.log("[Vymera] Starting wallet connection process...");
        console.log("[Vymera] Window.ethereum available?", !!window.ethereum);
        await this.connectWallet();
        console.log("[Vymera] Wallet connected successfully, refreshing UI...");
        container.innerHTML = "";
        this.createConnectedWalletUI(container);
      } catch (error) {
        console.error("[Vymera] Wallet connection failed:", error);
        connectButton.disabled = false;
        connectButton.textContent = "Connect MetaMask";
        connectButton.style.background = "#F6851B";
        const errorMessage = error.message || "Failed to connect wallet. Please try again.";
        const errorDiv = document.createElement("div");
        errorDiv.style.cssText = `
                    background: #ffebee;
                    border: 1px solid #f44336;
                    border-radius: 8px;
                    padding: 12px;
                    margin-top: 12px;
                    color: #c62828;
                    font-size: 14px;
                    text-align: left;
                `;
        errorDiv.innerHTML = `
                    <strong>Connection Failed:</strong><br>
                    ${errorMessage}<br><br>
                    <strong>Please try:</strong><br>
                    • Make sure MetaMask is unlocked<br>
                    • Refresh the page and try again<br>
                    • Check if MetaMask is enabled for this site
                `;
        const existingError = container.querySelector(".connection-error");
        if (existingError) {
          existingError.remove();
        }
        errorDiv.className = "connection-error";
        container.appendChild(errorDiv);
        setTimeout(() => {
          if (errorDiv.parentNode) {
            errorDiv.remove();
          }
        }, 1e4);
      }
    });
    connectButton.addEventListener("mouseenter", () => {
      if (!connectButton.disabled) {
        connectButton.style.background = "#E5740A";
      }
    });
    connectButton.addEventListener("mouseleave", () => {
      if (!connectButton.disabled) {
        connectButton.style.background = "#F6851B";
      }
    });
    promptSection.appendChild(metamaskIcon);
    promptSection.appendChild(title);
    promptSection.appendChild(description);
    promptSection.appendChild(connectButton);
    container.appendChild(promptSection);
  }
  createConnectedWalletUI(container) {
    console.log("[Vymera] Creating connected wallet UI...");
    const header = document.createElement("div");
    header.style.cssText = `
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 22px;
            margin-bottom: 16px;
            font-weight: 600;
            justify-content: space-between;
        `;
    const metamaskIcon = document.createElement("div");
    metamaskIcon.style.cssText = `
            width: 24px;
            height: 24px;
            background: #F6851B;
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 12px;
        `;
    metamaskIcon.textContent = "\uD83E\uDD8A";
    const headerText = document.createElement("span");
    headerText.textContent = "Pay with";
    const metamaskText = document.createElement("img");
    metamaskText.src = "https://freelogopng.com/images/all_img/1683020772metamask-logo-png.png";
    metamaskText.style.cssText = "height: 20px;";
    header.appendChild(headerText);
    header.appendChild(metamaskText);
    const walletInfo = document.createElement("div");
    walletInfo.style.cssText = `
            background: #f0f0f0;
            border-radius: 6px;
            padding: 8px 12px;
            margin-bottom: 16px;
            font-size: 12px;
            color: #666;
            display: flex;
            justify-content: space-between;
            align-items: center;
        `;
    if (this.connectedAccount) {
      const addressText = document.createElement("span");
      addressText.textContent = `Connected: ${this.connectedAccount.substring(0, 6)}...${this.connectedAccount.substring(38)}`;
      const connectedIcon = document.createElement("span");
      connectedIcon.style.cssText = "color: #4CAF50; font-weight: bold;";
      connectedIcon.textContent = "●";
      walletInfo.appendChild(addressText);
      walletInfo.appendChild(connectedIcon);
    } else {
      this.checkWalletConnection().then(async (isConnected) => {
        if (isConnected && this.connectedAccount) {
          walletInfo.innerHTML = `
                        <span>Connected: ${this.connectedAccount.substring(0, 6)}...${this.connectedAccount.substring(38)}</span>
                        <span style="color: #4CAF50; font-weight: bold;">●</span>
                    `;
        }
      });
    }
    const tokenDropdown = document.createElement("div");
    tokenDropdown.className = "vymera-token-dropdown";
    tokenDropdown.style.cssText = `
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            padding: 12px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 12px;
            background: white;
            transition: border-color 0.2s ease;
        `;
    const tokenIcon = document.createElement("img");
    tokenIcon.style.cssText = `
            width: 32px;
            height: 32px;
            border-radius: 50%;
            object-fit: contain;
        `;
    tokenIcon.src = "https://bscscan.com/assets/bsc/images/svg/logos/token-light.svg?v=25.9.4.0";
    tokenIcon.alt = "BNB Token";
    tokenIcon.onerror = () => {
      console.log("[Vymera] Failed to load default token image");
      tokenIcon.style.display = "none";
      const fallbackIcon = document.createElement("div");
      fallbackIcon.style.cssText = `
                width: 32px;
                height: 32px;
                background: #297561;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 16px;
                font-weight: bold;
                color: white;
            `;
      fallbackIcon.textContent = "B";
      tokenIcon.parentNode.replaceChild(fallbackIcon, tokenIcon);
    };
    const tokenInfo = document.createElement("div");
    tokenInfo.style.cssText = "flex: 1;";
    const tokenName = document.createElement("div");
    tokenName.textContent = "Binance Token";
    tokenName.style.cssText = "font-weight: 500;";
    const tokenBalance = document.createElement("div");
    tokenBalance.textContent = "Loading balance...";
    tokenBalance.style.cssText = "font-size: 12px; color: #666;";
    tokenInfo.appendChild(tokenName);
    tokenInfo.appendChild(tokenBalance);
    const dropdownArrow = document.createElement("div");
    dropdownArrow.textContent = "▼";
    dropdownArrow.style.cssText = "color: #666; font-size: 12px;";
    tokenDropdown.appendChild(tokenIcon);
    tokenDropdown.appendChild(tokenInfo);
    tokenDropdown.appendChild(dropdownArrow);
    tokenBalance.textContent = "Calculating amount needed...";
    this.calculateBNBAmount().then((calculation) => {
      this.fetchTokenBalances().then((balances2) => {
        const balanceText = `Balance: ${balances2.BNB} BNB`;
        const needsText = `Needs: ${calculation.bnbAmount} BNB`;
        tokenBalance.innerHTML = `${needsText}<br><small style="color: #999;">${balanceText}</small>`;
        const selectedToken = tokens.find((t) => t.symbol === "BNB");
        if (selectedToken) {
          const optionBalance = tokenOptions.children[0]?.querySelector(".token-balance");
          if (optionBalance) {
            optionBalance.innerHTML = `${needsText}<br><small style="color: #999;">${balanceText}</small>`;
          }
        }
      }).catch((error) => {
        console.error("[Vymera] Failed to fetch balances:", error);
        const needsText = `Needs: ${calculation.bnbAmount} BNB`;
        tokenBalance.textContent = needsText;
      });
    }).catch((error) => {
      console.error("[Vymera] Failed to calculate BNB amount:", error);
      tokenBalance.textContent = "Needs: 0.2084 BNB (Fallback)";
      this.fetchTokenBalances().then((balances2) => {
        const balanceText = `Balance: ${balances2.BNB} BNB`;
        const needsText = `Needs: 0.2084 BNB (Fallback)`;
        tokenBalance.innerHTML = `${needsText}<br><small style="color: #999;">${balanceText}</small>`;
      }).catch(() => {
        tokenBalance.textContent = "Needs: 0.2084 BNB (Fallback)";
      });
    });
    const tokenOptions = document.createElement("div");
    tokenOptions.className = "vymera-token-options";
    tokenOptions.style.cssText = `
            display: none;
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            z-index: 1000;
            margin-top: 4px;
        `;
    const tokens = [
      { symbol: "BNB", name: "Binance Token", icon: "B", color: "#297561", address: "0xbb4CdB9Bd36B01bD1cBaEBF2De08d9173bc095c", enabled: true, imageUrl: "https://bscscan.com/assets/bsc/images/svg/logos/token-light.svg?v=25.9.4.0", chainId: "0x38", chainName: "BNB Smart Chain" },
      { symbol: "TAC", name: "Toncoin Access Chain", icon: "T", color: "#0088CC", address: "0x76A797A59Ba2C17726896976B7B4E3fA56eD3B51", enabled: true, imageUrl: "https://s2.coinmarketcap.com/static/img/coins/64x64/37338.png", chainId: "0xef", chainName: "TAC Mainnet" },
      { symbol: "CELO", name: "Celo", icon: "C", color: "#35D07F", address: "0x88eeC49252c8cbc039DCdB394c0c2BA2f1637EA0", enabled: true, imageUrl: "https://static1.tokenterminal.com//celo/logo.png?logo_hash=e37b727d573d56157f3d2373da11d4450a43ba1f", chainId: "0xa4ec", chainName: "Celo Mainnet" },
      { symbol: "GLMR", name: "Moonbeam", icon: "M", color: "#53CBC9", address: "0xAcc15dC74880C9944775448304B263D191c6077F", enabled: true, imageUrl: "https://assets.kraken.com/marketing/web/icons-uni-webp/s_glmr.webp?i=kds", chainId: "0x504", chainName: "Moonbeam" }
    ];
    let balances = {};
    this.fetchTokenBalances().then((fetchedBalances) => {
      balances = fetchedBalances;
      tokens.forEach((token, index) => {
        const optionBalance = tokenOptions.children[index]?.querySelector(".token-balance");
        if (optionBalance) {
          optionBalance.textContent = `Balance: ${balances[token.symbol] || "0.0000"} ${token.symbol}`;
        }
      });
    });
    tokens.forEach((token) => {
      const option = document.createElement("div");
      option.className = "vymera-token-option";
      option.style.cssText = `
                padding: 12px;
                cursor: ${token.enabled ? "pointer" : "not-allowed"};
                display: flex;
                align-items: center;
                gap: 12px;
                transition: background-color 0.2s ease;
                opacity: ${token.enabled ? "1" : "0.5"};
                position: relative;
            `;
      const optionIcon = token.imageUrl ? document.createElement("img") : document.createElement("div");
      if (token.imageUrl) {
        optionIcon.style.cssText = `
                    width: 32px;
                    height: 32px;
                    border-radius: 50%;
                    object-fit: contain;
                `;
        optionIcon.src = token.imageUrl;
        optionIcon.alt = `${token.symbol} Token`;
        console.log(`[Vymera] Loading image for ${token.symbol}:`, token.imageUrl);
        optionIcon.onload = () => {
          console.log(`[Vymera] Successfully loaded image for ${token.symbol}`);
        };
        optionIcon.onerror = () => {
          console.log(`[Vymera] Failed to load image for ${token.symbol}:`, token.imageUrl);
          optionIcon.style.display = "none";
          const fallbackIcon = document.createElement("div");
          fallbackIcon.style.cssText = `
                        width: 32px;
                        height: 32px;
                        background: ${token.color};
                        border-radius: 50%;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        font-size: 16px;
                        font-weight: bold;
                        color: white;
                    `;
          fallbackIcon.textContent = token.icon;
          optionIcon.parentNode.replaceChild(fallbackIcon, optionIcon);
        };
      } else {
        optionIcon.style.cssText = `
                    width: 32px;
                    height: 32px;
                    background: ${token.enabled ? token.color : "#cccccc"};
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-size: 16px;
                    font-weight: bold;
                    color: white;
                `;
        optionIcon.textContent = token.icon;
      }
      const optionInfo = document.createElement("div");
      optionInfo.style.cssText = "flex: 1;";
      const optionHeader = document.createElement("div");
      optionHeader.style.cssText = "display: flex; justify-content: space-between; align-items: center;";
      const optionNameContainer = document.createElement("div");
      const optionName = document.createElement("div");
      optionName.textContent = token.name;
      optionName.style.cssText = `font-weight: 500; color: ${token.enabled ? "#333" : "#999"};`;
      optionNameContainer.appendChild(optionName);
      const optionRight = document.createElement("div");
      optionRight.style.cssText = "text-align: right;";
      if (!token.enabled) {
        const comingSoon = document.createElement("div");
        comingSoon.textContent = "Coming soon";
        comingSoon.style.cssText = "font-size: 11px; color: #297561; font-style: italic; margin-bottom: 2px;";
        optionRight.appendChild(comingSoon);
      }
      const optionBalance = document.createElement("div");
      optionBalance.className = "token-balance";
      optionBalance.textContent = "Loading...";
      optionBalance.style.cssText = `font-size: 12px; color: ${token.enabled ? "#666" : "#999"}; font-weight: normal;`;
      optionRight.appendChild(optionBalance);
      optionHeader.appendChild(optionNameContainer);
      optionHeader.appendChild(optionRight);
      const optionSymbol = document.createElement("div");
      optionSymbol.textContent = token.symbol;
      optionSymbol.style.cssText = `font-size: 12px; color: ${token.enabled ? "#666" : "#999"};`;
      optionInfo.appendChild(optionHeader);
      optionInfo.appendChild(optionSymbol);
      option.appendChild(optionIcon);
      option.appendChild(optionInfo);
      option.addEventListener("click", async (e) => {
        if (!token.enabled) {
          e.stopPropagation();
          return;
        }
        if (token.chainId) {
          try {
            await this.switchToBlockchain(token.chainId, token.chainName);
          } catch (error) {
            console.error(`[Vymera] Failed to switch to ${token.chainName}:`, error);
          }
        }
        if (token.imageUrl) {
          tokenIcon.src = token.imageUrl;
          tokenIcon.alt = `${token.symbol} Token`;
        } else {
          tokenIcon.textContent = token.icon;
          tokenIcon.style.background = token.color;
        }
        tokenName.textContent = token.name;
        tokenBalance.textContent = `Balance: ${balances[token.symbol] || "0.0000"} ${token.symbol}`;
        tokenOptions.style.display = "none";
      });
      option.addEventListener("mouseenter", () => {
        if (token.enabled) {
          option.style.backgroundColor = "#f5f5f5";
        }
      });
      option.addEventListener("mouseleave", () => {
        if (token.enabled) {
          option.style.backgroundColor = "transparent";
        }
      });
      tokenOptions.appendChild(option);
    });
    tokenDropdown.style.position = "relative";
    tokenDropdown.appendChild(tokenOptions);
    tokenDropdown.addEventListener("click", (e) => {
      e.stopPropagation();
      const isOpen = tokenOptions.style.display !== "none";
      tokenOptions.style.display = isOpen ? "none" : "block";
      dropdownArrow.textContent = isOpen ? "▼" : "▲";
    });
    document.addEventListener("click", () => {
      tokenOptions.style.display = "none";
      dropdownArrow.textContent = "▼";
    });
    const paymentSummary = document.createElement("div");
    paymentSummary.className = "vymera-payment-summary";
    paymentSummary.style.cssText = `
            background: #f8f9fa;
            border-radius: 8px;
            padding: 16px;
            margin-top: 16px;
            border: 1px solid #e9ecef;
        `;
    paymentSummary.innerHTML = `
            <div style="font-weight: 600; font-size: 16px; color: #495057; margin-bottom: 12px;">Payment Summary</div>
            <div id="payment-calculation" style="font-size: 12px; color: #666;">
                Calculating payment amount...
            </div>
        `;
    setTimeout(() => {
      this.calculateBNBAmount().then((calculation) => {
        const summaryElement = paymentSummary.querySelector("#payment-calculation");
        if (summaryElement) {
          summaryElement.innerHTML = `
                    <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                        <span>Booking Total:</span>
                        <span style="font-weight: 500;">$${calculation.totalPrice.amount.toFixed(2)} ${calculation.totalPrice.currency}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                        <span>BNB Rate:</span>
                        <span style="font-weight: 500;">1 BNB = $${calculation.exchangeRate.toFixed(2)} ${calculation.totalPrice.currency}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding-top: 16px; font-size:16px; border-top: 1px solid #dee2e6; font-weight: 600; color: #495057;">
                        <span>You Pay:</span>
                        <span style="color: #297561;">${calculation.bnbAmount} BNB</span>
                    </div>
                `;
        }
      }).catch((error) => {
        console.error("[Vymera] Failed to calculate payment summary:", error);
        const summaryElement = paymentSummary.querySelector("#payment-calculation");
        if (summaryElement) {
          summaryElement.innerHTML = `
                    <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                        <span>Booking Total:</span>
                        <span style="font-weight: 500;">125.12 SGD</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                        <span>BNB Rate:</span>
                        <span style="font-weight: 500;">1 BNB = 600.00 USD (Demo)</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding-top: 8px; border-top: 1px solid #dee2e6; font-weight: 600; color: #495057;">
                        <span>You Pay:</span>
                        <span style="color: #297561;">0.2084 BNB</span>
                    </div>
                `;
        }
      }).catch((error) => {
        console.error("[Vymera] Failed to calculate payment summary:", error);
        const summaryElement = paymentSummary.querySelector("#payment-calculation");
        if (summaryElement) {
          summaryElement.innerHTML = `
                    <div style="color: #666; text-align: center;">
                        Failed to calculate payment. Using demo values:<br>
                        <strong style="color: #297561;">0.2084 BNB for 125.12 SGD</strong>
                    </div>
                `;
        }
      });
    }, 1500);
    const payButton = document.createElement("button");
    payButton.style.cssText = `
            width: 100%;
            background: #297561;
            color: #fff;
            border: none;
            border-radius: 8px;
            padding: 16px;
            margin-top: 24px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        `;
    payButton.innerHTML = `Pay with Crypto`;
    payButton.addEventListener("mouseenter", () => {
      payButton.style.background = "#1F5445";
    });
    payButton.addEventListener("mouseleave", () => {
      payButton.style.background = "#297561";
    });
    payButton.addEventListener("click", async (e) => {
      e.preventDefault();
      e.stopPropagation();
      payButton.disabled = true;
      payButton.innerHTML = "<span>Processing...</span>";
      payButton.style.background = "#ccc";
      try {
        await this.handleCryptoPayment();
      } catch (error) {
        console.error("[Vymera] Payment failed:", error);
        payButton.disabled = false;
        payButton.innerHTML = `
                    <span style="font-size: 20px;">₿</span>
                    Pay with Crypto
                `;
        payButton.style.background = "#297561";
      }
    });
    container.appendChild(header);
    container.appendChild(walletInfo);
    container.appendChild(tokenDropdown);
    container.appendChild(paymentSummary);
    container.appendChild(payButton);
  }
  hideCryptoPaymentOptions(paymentSection) {
    if (paymentSection.dataset.originalContent) {
      paymentSection.innerHTML = paymentSection.dataset.originalContent;
    }
    console.log("[Vymera] Crypto payment options hidden, original payment section restored");
  }
  async fetchBNBPrice() {
    console.log("[Vymera] Fetching BNB price...");
    return new Promise((resolve, reject) => {
      const handleResponse = (event) => {
        console.log("[Vymera] Received BNB price response:", event.detail);
        window.removeEventListener("vymera-price-response", handleResponse);
        if (event.detail.success) {
          resolve(event.detail.prices);
        } else {
          console.error("[Vymera] BNB price fetch failed, using fallback:", event.detail.error);
          resolve(event.detail.prices);
        }
      };
      window.addEventListener("vymera-price-response", handleResponse);
      console.log("[Vymera] Dispatching BNB price fetch request...");
      window.dispatchEvent(new CustomEvent("vymera-fetch-bnb-price"));
      setTimeout(() => {
        window.removeEventListener("vymera-price-response", handleResponse);
        console.log("[Vymera] BNB price fetch timeout, using fallback prices");
        resolve({ usd: 600, sgd: 810 });
      }, 3000);
    });
  }
  extractTotalPrice() {
    console.log("[Vymera] Extracting booking price from page...");
    const priceSelectors = [
      '[data-testid="book-it-default"] span:contains("Total")',
      '[data-testid="book-it-default"] div:contains("SGD")',
      '[data-testid="book-it-default"] div:contains("USD")',
      '[data-testid="book-it-default"] div:contains("$")',
      'div[aria-label*="price"]',
      'div[class*="price-item"] span:contains("Total")',
      'div[class*="total"] span:contains("$")',
      'span:contains("Total")',
      'div:contains("Total") span:contains("$")'
    ];
    const priceElements = document.querySelectorAll("span, div");
    let totalPrice = 0;
    let currency = "USD";
    for (const element of priceElements) {
      const text = element.textContent?.trim() || "";
      const priceMatch = text.match(/(?:SGD\s*)?(?:\$|USD\s*)?(\d{1,3}(?:,\d{3})*(?:\.\d{2})?)\s*(?:SGD|USD)?/i);
      if (priceMatch && text.toLowerCase().includes("total")) {
        const amountStr = priceMatch[1].replace(/,/g, "");
        const amount = parseFloat(amountStr);
        if (amount > totalPrice) {
          totalPrice = amount;
          if (text.includes("SGD") || text.includes("S$")) {
            currency = "SGD";
          } else if (text.includes("USD") || text.includes("US$")) {
            currency = "USD";
          }
        }
      }
    }
    if (totalPrice === 0) {
      const pageText = document.body.textContent || "";
      const fallbackMatch = pageText.match(/Total[^$]*\$(\d{1,3}(?:,\d{3})*(?:\.\d{2})?)\s*(SGD|USD)?/i);
      if (fallbackMatch) {
        totalPrice = parseFloat(fallbackMatch[1].replace(/,/g, ""));
        currency = fallbackMatch[2] || "SGD";
      }
    }
    console.log("[Vymera] Extracted price:", totalPrice, currency);
    return { amount: totalPrice || 125.12, currency: currency.toUpperCase() };
  }
  async calculateBNBAmount() {
    try {
      const totalPrice = this.extractTotalPrice();
      console.log("[Vymera] Total booking price:", totalPrice);
      const bnbPrices = await this.fetchBNBPrice();
      console.log("[Vymera] BNB prices:", bnbPrices);
      const exchangeRate = totalPrice.currency === "SGD" ? bnbPrices.sgd : bnbPrices.usd;
      const bnbAmount = (totalPrice.amount / exchangeRate).toFixed(6);
      console.log("[Vymera] Calculated BNB amount:", {
        totalPrice,
        exchangeRate,
        bnbAmount
      });
      return {
        bnbAmount,
        exchangeRate,
        totalPrice
      };
    } catch (error) {
      console.error("[Vymera] Error calculating BNB amount:", error);
      return {
        bnbAmount: "0.2084",
        exchangeRate: 600,
        totalPrice: { amount: 125.12, currency: "SGD" }
      };
    }
  }
  async generateFundingId() {
    console.log("[Vymera] Generating funding ID from backend...");
    try {
      const response = await fetch("http://127.0.0.1:8000/fundings/generate", {
        method: "POST",
        headers: {
          "X-API-Key": "test",
          "Content-Type": "application/json"
        }
      });
      if (!response.ok) {
        throw new Error(`Backend API error: ${response.status} ${response.statusText}`);
      }
      const data = await response.json();
      const fundingId = BigInt(data.id);
      console.log("[Vymera] Generated funding ID:", fundingId.toString());
      return fundingId;
    } catch (error) {
      console.error("[Vymera] Failed to generate funding ID:", error);
      console.log("[Vymera] Backend not available, using fallback ID generation");
      const fallbackId = BigInt(Date.now() * 1000 + Math.floor(Math.random() * 1000));
      console.log("[Vymera] Using fallback funding ID:", fallbackId.toString());
      return fallbackId;
    }
  }
  async pollForCardDetails(fundingId, processingMessage) {
    console.log("[Vymera] Starting to poll for card details...");
    const maxRetries = 10;
    const delayMs = 2000;
    for (let attempt = 1;attempt <= maxRetries; attempt++) {
      console.log(`[Vymera] Polling attempt ${attempt}/${maxRetries} for funding ID: ${fundingId}`);
      try {
        const response = await fetch(`http://127.0.0.1:8000/fundings/${fundingId}`, {
          method: "GET",
          headers: {
            "X-API-Key": "test",
            "Content-Type": "application/json"
          }
        });
        if (!response.ok) {
          throw new Error(`API error: ${response.status} ${response.statusText}`);
        }
        const data = await response.json();
        console.log("[Vymera] Funding details response:", data);
        if (data.card && data.card.token) {
          console.log("[Vymera] Card details received:", data.card);
          processingMessage.innerHTML = "✓ Card created!<br><small>Processing payment...</small>";
          await this.simulatePayment(fundingId, processingMessage);
          return;
        } else {
          console.log("[Vymera] Card not ready yet, waiting...");
          processingMessage.innerHTML = `✓ Transaction confirmed!<br><small>Waiting for card... (${attempt}/${maxRetries})</small>`;
        }
      } catch (error) {
        console.error(`[Vymera] Polling attempt ${attempt} failed:`, error);
        processingMessage.innerHTML = `✓ Transaction confirmed!<br><small>Retrying... (${attempt}/${maxRetries})</small>`;
      }
      if (attempt < maxRetries) {
        await new Promise((resolve) => setTimeout(resolve, delayMs));
      }
    }
    console.error("[Vymera] Failed to get card details after 10 attempts");
    processingMessage.innerHTML = "❌ Failed to create card<br><small>Please try again</small>";
    processingMessage.style.background = "rgba(244, 67, 54, 0.1)";
    processingMessage.style.borderColor = "rgba(244, 67, 54, 0.3)";
  }
  async simulatePayment(fundingId, processingMessage) {
    console.log("[Vymera] Processing payment for funding ID:", fundingId);
    try {
      const response = await fetch(`http://127.0.0.1:8000/fundings/${fundingId}/simulate`, {
        method: "POST",
        headers: {
          "X-API-Key": "test",
          "Content-Type": "application/json"
        },
        body: JSON.stringify({
          descriptor: "AIRBNB",
          mcc: "7011"
        })
      });
      if (!response.ok) {
        throw new Error(`Simulation API error: ${response.status} ${response.statusText}`);
      }
      const data = await response.json();
      console.log("[Vymera] Payment simulation response:", data);
      processingMessage.innerHTML = "✓ Payment completed!<br><small>Booking confirmed</small>";
      processingMessage.style.background = "rgba(76, 175, 80, 0.1)";
      processingMessage.style.borderColor = "rgba(76, 175, 80, 0.3)";
    } catch (error) {
      console.error("[Vymera] Payment simulation failed:", error);
      processingMessage.innerHTML = "❌ Payment simulation failed<br><small>Please try again</small>";
      processingMessage.style.background = "rgba(244, 67, 54, 0.1)";
      processingMessage.style.borderColor = "rgba(244, 67, 54, 0.3)";
    }
  }
  async processPayment() {}
  showPaymentSuccess() {
    const successMessage = document.createElement("div");
    successMessage.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: rgba(76, 175, 80, 0.1);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid rgba(76, 175, 80, 0.3);
            color: #333;
            padding: 20px 28px;
            border-radius: 20px;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.37);
            z-index: 10000;
            font-weight: 600;
            width: 300px;
            max-width: 300px;
            animation: slideIn 0.3s ease-out;
        `;
    successMessage.innerHTML = `
            <span style="font-size: 24px;">✅</span> Payment Successful!<br>
            <small style="opacity: 0.9;">Your BNB payment has been processed</small>
        `;
    document.body.appendChild(successMessage);
    setTimeout(() => {
      successMessage.style.animation = "slideIn 0.3s ease-out reverse";
      setTimeout(() => {
        successMessage.remove();
      }, 300);
    }, 5000);
  }
  destroy() {
    if (this.observer) {
      this.observer.disconnect();
      this.observer = null;
    }
  }
}
var injector = null;
window.addEventListener("beforeunload", () => {
  if (injector) {
    injector.destroy();
    injector = null;
  }
});
if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", () => {
    injector = new VymeraInjector;
  });
} else {
  injector = new VymeraInjector;
}
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.type === "PAYMENT_COMPLETE" || request.type === "PAYMENT_CONFIRMED") {
    console.log("[Vymera] Payment completed successfully");
    const successMessage = document.createElement("div");
    successMessage.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: rgba(76, 175, 80, 0.1);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid rgba(76, 175, 80, 0.3);
            color: #333;
            padding: 20px 28px;
            border-radius: 20px;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.37);
            z-index: 10000;
            font-weight: 600;
            animation: slideIn 0.3s ease-out;
            width: 300px;
            max-width: 300px;
        `;
    successMessage.innerHTML = "<span style='font-size: 20px;'>✓</span> Crypto payment successful!<br><small style='opacity: 0.9;'>Completing your booking...</small>";
    document.body.appendChild(successMessage);
    if (injector && injector.originalConfirmPayButton) {}
    setTimeout(() => {
      successMessage.remove();
    }, 5000);
    sendResponse({ success: true });
  }
  return true;
});
