// extension/background/background.ts
class VymeraBackground {
  currentPayment = null;
  offscreenDocumentPath = "offscreen.html";
  backendUrl = "http://localhost:3000";
  constructor() {
    this.setupListeners();
  }
  setupListeners() {
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      console.log("[Background] Received message:", request.type);
      switch (request.type) {
        case "INITIATE_BNB_PAYMENT":
        case "OPEN_PAYMENT_POPUP":
          this.handlePaymentInitiation(request.data, sender.tab?.id);
          sendResponse({ success: true });
          break;
        case "CONNECT_WALLET_FROM_POPUP":
          this.handleWalletConnectionFromPopup(sender.tab?.id);
          sendResponse({ success: true });
          break;
        case "PAYMENT_CONFIRMED":
          this.handlePaymentConfirmed(request.data);
          sendResponse({ success: true });
          break;
        case "GET_PAYMENT_DATA":
          sendResponse({ data: this.currentPayment });
          break;
        case "CARD_CREATED":
          this.handleCardCreated(request.data);
          sendResponse({ success: true });
          break;
        case "CARD_ADDED_TO_AIRBNB":
          this.handleCardAddedToAirbnb(request.data);
          sendResponse({ success: true });
          break;
        default:
          console.warn("[Background] Unknown message type:", request.type);
      }
      return true;
    });
    chrome.action.onClicked.addListener((tab) => {
      if (tab.url?.includes("airbnb.com")) {
        chrome.tabs.sendMessage(tab.id, { type: "CHECK_CHECKOUT_PAGE" });
      }
    });
  }
  async handlePaymentInitiation(paymentData, tabId) {
    this.currentPayment = paymentData;
    await chrome.storage.local.set({
      currentPayment: paymentData,
      paymentTabId: tabId
    });
    const popup = await chrome.windows.create({
      url: chrome.runtime.getURL("popup.html"),
      type: "popup",
      width: 420,
      height: 600,
      left: 100,
      top: 100
    });
    await chrome.storage.local.set({ popupWindowId: popup.id });
  }
  async handleWalletConnectionFromPopup(tabId) {
    const { paymentTabId } = await chrome.storage.local.get("paymentTabId");
    const targetTabId = tabId || paymentTabId;
    if (!targetTabId) {
      console.error("[Background] No tab ID available for wallet connection");
      return;
    }
    chrome.tabs.sendMessage(targetTabId, {
      type: "CONNECT_WALLET_REQUEST",
      data: {}
    }, (response) => {
      if (response?.success && response.address) {
        chrome.storage.local.set({
          connectedAddress: response.address,
          walletConnected: true
        });
        chrome.runtime.sendMessage({
          type: "WALLET_CONNECTED",
          data: { address: response.address }
        });
      }
    });
  }
  async handlePaymentConfirmed(data) {
    console.log("[Background] Payment confirmed:", data);
    await this.waitForConfirmations(data.txHash);
    try {
      const response = await fetch(`${this.backendUrl}/api/create-virtual-card`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json"
        },
        body: JSON.stringify({
          payment: this.currentPayment,
          transaction: data
        })
      });
      if (!response.ok) {
        throw new Error(`Failed to create virtual card: ${response.statusText}`);
      }
      const virtualCard = await response.json();
      console.log("[Background] Virtual card created:", virtualCard);
      await chrome.storage.local.set({
        virtualCard,
        cardCreatedAt: Date.now()
      });
      await this.addCardToAirbnb(virtualCard);
    } catch (error) {
      console.error("[Background] Error creating virtual card:", error);
      this.notifyError("Failed to create virtual card");
      const { popupWindowId } = await chrome.storage.local.get("popupWindowId");
      if (popupWindowId) {
        setTimeout(() => {
          chrome.windows.remove(popupWindowId);
        }, 1500);
      }
    }
  }
  async waitForConfirmations(txHash, requiredConfirmations = 3) {
    console.log("[Background] Waiting for confirmations:", txHash);
    return new Promise((resolve, reject) => {
      const checkConfirmations = async () => {
        try {
          const response = await fetch(`${this.backendUrl}/api/transaction/${txHash}/confirmations`);
          if (response.ok) {
            const data = await response.json();
            console.log("[Background] Confirmations:", data.confirmations);
            if (data.confirmations >= requiredConfirmations) {
              console.log("[Background] Transaction confirmed with", data.confirmations, "confirmations");
              resolve();
              return;
            }
          }
          setTimeout(checkConfirmations, 15000);
        } catch (error) {
          console.error("[Background] Error checking confirmations:", error);
          setTimeout(checkConfirmations, 15000);
        }
      };
      checkConfirmations();
      setTimeout(() => {
        reject(new Error("Timeout waiting for confirmations"));
      }, 600000);
    });
  }
  async addCardToAirbnb(virtualCard) {
    await chrome.offscreen.createDocument({
      url: this.offscreenDocumentPath,
      reasons: ["DOM_SCRAPING"],
      justification: "Adding virtual card to Airbnb payment methods"
    });
    const response = await chrome.runtime.sendMessage({
      type: "ADD_CARD_TO_AIRBNB",
      target: "offscreen",
      data: {
        card: virtualCard,
        paymentTabId: (await chrome.storage.local.get("paymentTabId")).paymentTabId
      }
    });
    if (response?.success) {
      console.log("[Background] Card successfully added to Airbnb");
    } else {
      throw new Error("Failed to add card to Airbnb");
    }
  }
  async handleCardCreated(data) {
    console.log("[Background] Virtual card created successfully");
    const { popupWindowId } = await chrome.storage.local.get("popupWindowId");
    if (popupWindowId) {
      chrome.windows.remove(popupWindowId);
    }
    await this.addCardToAirbnb(data.card);
  }
  async handleCardAddedToAirbnb(data) {
    if (data.success) {
      console.log("[Background] Card added to Airbnb successfully");
      const { paymentTabId } = await chrome.storage.local.get("paymentTabId");
      if (paymentTabId) {
        chrome.tabs.sendMessage(paymentTabId, {
          type: "PAYMENT_COMPLETE",
          data: { success: true }
        });
      }
      await chrome.storage.local.remove([
        "currentPayment",
        "virtualCard",
        "paymentTabId",
        "popupWindowId"
      ]);
      await chrome.offscreen.closeDocument();
    } else {
      this.notifyError("Failed to add card to Airbnb");
    }
  }
  notifyError(message) {
    chrome.notifications.create({
      type: "basic",
      iconUrl: chrome.runtime.getURL("assets/icon128.png"),
      title: "Vymera Error",
      message,
      priority: 2
    });
  }
}
new VymeraBackground;
chrome.runtime.onInstalled.addListener(() => {
  console.log("[Vymera] Extension installed");
});
setInterval(() => {
  chrome.storage.local.get(["keepAlive"], (result) => {
    chrome.storage.local.set({ keepAlive: Date.now() });
  });
}, 20000);
